<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Exam;
use App\Models\InviteCode;
use App\Models\Student;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CodeController extends Controller
{
    public function index(Request $request, Exam $exam)
    {
        $q = trim((string)$request->query('q', ''));
        $codes = InviteCode::query()
            ->where('exam_id', $exam->id)
            ->when($q !== '', function ($qq) use ($q) {
                $qq->where('code', 'like', "%{$q}%")
                   ->orWhere('student_snapshot_name', 'like', "%{$q}%")
                   ->orWhere('student_snapshot_code', 'like', "%{$q}%");
            })
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        $students = Student::query()->whereNull('deleted_at')->orderBy('full_name')->limit(200)->get();

        return view('admin.codes.index', compact('exam', 'codes', 'q', 'students'));
    }

    public function generate(Request $request, Exam $exam)
    {
        $data = $request->validate([
            'count' => 'required|integer|min:1|max:500',
            'expires_hours' => 'required|integer|min:1|max:720',
            'student_ids' => 'nullable|array',
            'student_ids.*' => 'integer',
        ]);

        $expiresAt = now()->addHours((int)$data['expires_hours']);
        $count = (int)$data['count'];

        $studentIds = collect($data['student_ids'] ?? [])->map(fn($v)=>(int)$v)->filter()->values();

        $created = 0;
        for ($i = 0; $i < $count; $i++) {
            $code = null;
            // ensure unique
            do {
                $code = strtoupper(Str::random(8));
            } while (InviteCode::query()->where('code', $code)->exists());

            $student = null;
            if ($studentIds->isNotEmpty()) {
                $sid = $studentIds[$i % $studentIds->count()];
                $student = Student::query()->whereKey($sid)->first();
            }

            InviteCode::create([
                'exam_id' => $exam->id,
                'student_id' => $student?->id,
                'code' => $code,
                'expires_at' => $expiresAt,
                'student_snapshot_name' => $student?->full_name,
                'student_snapshot_code' => $student?->student_code,
            ]);

            $created++;
        }

        return back()->with('ok', "{$created} کد ایجاد شد.");
    }

    public function destroy(InviteCode $code)
    {
        $code->delete();
        return back()->with('ok', 'کد حذف شد (حذف نرم).');
    }
}
