<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Exam;
use Illuminate\Http\Request;

class ExamController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string)$request->query('q', ''));
        $exams = Exam::query()
            ->when($q !== '', fn($qq) => $qq->where('title', 'like', "%{$q}%"))
            ->orderByDesc('id')
            ->paginate(15)
            ->withQueryString();

        return view('admin.exams.index', compact('exams', 'q'));
    }

    public function create()
    {
        return view('admin.exams.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'duration_minutes' => 'required|integer|min:1|max:600',
            'pass_threshold' => 'required|integer|min:0|max:100',
            'start_at' => 'nullable|date',
            'end_at' => 'nullable|date|after_or_equal:start_at',
            'show_answers_after_minutes' => 'nullable|integer|min:0|max:100000',
            'is_active' => 'nullable|boolean',
        ]);

        $data['is_active'] = (bool)($request->boolean('is_active', true));

        $exam = Exam::create($data);

        return redirect()->route('admin.exams.edit', $exam)->with('ok', 'آزمون ایجاد شد.');
    }

    public function edit(Exam $exam)
    {
        return view('admin.exams.edit', compact('exam'));
    }

    public function update(Request $request, Exam $exam)
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'duration_minutes' => 'required|integer|min:1|max:600',
            'pass_threshold' => 'required|integer|min:0|max:100',
            'start_at' => 'nullable|date',
            'end_at' => 'nullable|date|after_or_equal:start_at',
            'show_answers_after_minutes' => 'nullable|integer|min:0|max:100000',
            'is_active' => 'nullable|boolean',
        ]);

        $data['is_active'] = (bool)($request->boolean('is_active', false));

        $exam->update($data);

        return back()->with('ok', 'ذخیره شد.');
    }

    public function destroy(Exam $exam)
    {
        $exam->delete();
        return redirect()->route('admin.exams.index')->with('ok', 'آزمون حذف شد (حذف نرم).');
    }

    public function show(Exam $exam)
    {
        return redirect()->route('admin.exams.edit', $exam);
    }
}
