<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Exam;
use App\Models\Question;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class QuestionController extends Controller
{
    public function index(Request $request, Exam $exam)
    {
        $q = trim((string)$request->query('q', ''));
        $questions = Question::query()
            ->where('exam_id', $exam->id)
            ->when($q !== '', fn($qq) => $qq->where('question_text', 'like', "%{$q}%"))
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        return view('admin.questions.index', compact('exam', 'questions', 'q'));
    }

    public function store(Request $request, Exam $exam)
    {
        $data = $request->validate([
            'question_text' => 'required|string',
            'choice0' => 'required|string|max:500',
            'choice1' => 'required|string|max:500',
            'choice2' => 'required|string|max:500',
            'choice3' => 'required|string|max:500',
            'correct_index' => 'required|integer|min:0|max:3',
            'score' => 'required|integer|min:1|max:100',
            'is_active' => 'nullable|boolean',
        ]);

        Question::create([
            'exam_id' => $exam->id,
            'question_text' => $data['question_text'],
            'choices' => [$data['choice0'], $data['choice1'], $data['choice2'], $data['choice3']],
            'correct_index' => (int)$data['correct_index'],
            'score' => (int)$data['score'],
            'is_active' => (bool)$request->boolean('is_active', true),
        ]);

        return back()->with('ok', 'سوال اضافه شد.');
    }

    public function update(Request $request, Question $question)
    {
        $data = $request->validate([
            'question_text' => 'required|string',
            'choice0' => 'required|string|max:500',
            'choice1' => 'required|string|max:500',
            'choice2' => 'required|string|max:500',
            'choice3' => 'required|string|max:500',
            'correct_index' => 'required|integer|min:0|max:3',
            'score' => 'required|integer|min:1|max:100',
        ]);

        $question->update([
            'question_text' => $data['question_text'],
            'choices' => [$data['choice0'], $data['choice1'], $data['choice2'], $data['choice3']],
            'correct_index' => (int)$data['correct_index'],
            'score' => (int)$data['score'],
        ]);

        return back()->with('ok', 'ویرایش شد.');
    }

    public function toggle(Request $request, Question $question)
    {
        $question->is_active = !$question->is_active;
        $question->save();

        return back()->with('ok', $question->is_active ? 'سوال فعال شد.' : 'سوال غیرفعال شد.');
    }

    public function uploadImage(Request $request, Question $question)
    {
        $request->validate([
            'image' => 'required|image|max:4096',
        ]);

        $path = $request->file('image')->store('question-images', 'public');
        // delete old
        if ($question->image_path) {
            Storage::disk('public')->delete($question->image_path);
        }
        $question->image_path = $path;
        $question->save();

        return back()->with('ok', 'تصویر ثبت شد.');
    }

    public function destroy(Question $question)
    {
        $question->delete();
        return back()->with('ok', 'سوال حذف شد (حذف نرم).');
    }
}
