<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Student;
use Illuminate\Http\Request;

class StudentController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string)$request->query('q', ''));
        $students = Student::query()
            ->when($q !== '', function ($qq) use ($q) {
                $qq->where('full_name', 'like', "%{$q}%")
                   ->orWhere('student_code', 'like', "%{$q}%");
            })
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        return view('admin.students.index', compact('students', 'q'));
    }

    public function create()
    {
        return view('admin.students.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'student_code' => 'required|string|max:64|unique:students,student_code',
            'full_name' => 'required|string|max:255',
            'class_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:50',
            'is_active' => 'nullable|boolean',
        ]);

        $data['is_active'] = (bool)$request->boolean('is_active', true);

        Student::create($data);

        return redirect()->route('admin.students.index')->with('ok', 'دانش‌آموز اضافه شد.');
    }

    public function edit(Student $student)
    {
        return view('admin.students.edit', compact('student'));
    }

    public function update(Request $request, Student $student)
    {
        $data = $request->validate([
            'student_code' => 'required|string|max:64|unique:students,student_code,' . $student->id,
            'full_name' => 'required|string|max:255',
            'class_name' => 'nullable|string|max:255',
            'phone' => 'nullable|string|max:50',
            'is_active' => 'nullable|boolean',
        ]);

        $data['is_active'] = (bool)$request->boolean('is_active', false);

        $student->update($data);

        return back()->with('ok', 'ذخیره شد.');
    }

    public function destroy(Student $student)
    {
        $student->delete();
        return redirect()->route('admin.students.index')->with('ok', 'دانش‌آموز حذف شد (حذف نرم).');
    }

    public function show(Student $student)
    {
        return redirect()->route('admin.students.edit', $student);
    }
}
