<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Models\AntiCheatSetting;
use App\Models\Attempt;
use App\Models\Answer;
use App\Models\InviteCode;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TakeExamController extends Controller
{
    public function enterCode()
    {
        return view('student.enter_code');
    }

    public function postEnterCode(Request $request)
    {
        $data = $request->validate(['code' => 'required|string']);
        $codeStr = strtoupper(trim($data['code']));

        $invite = InviteCode::query()
            ->with('exam')
            ->where('code', $codeStr)
            ->whereNull('deleted_at')
            ->first();

        if (!$invite || !$invite->exam || $invite->exam->deleted_at || !$invite->exam->is_active) {
            return back()->withErrors(['code' => 'کد نامعتبر است.']);
        }
        if ($invite->used_at) return back()->withErrors(['code' => 'این کد قبلاً استفاده شده است.']);
        if (now()->greaterThan($invite->expires_at)) return back()->withErrors(['code' => 'کد منقضی شده است.']);

        return view('student.preview', [
            'invite' => $invite,
            'exam' => $invite->exam,
        ]);
    }

    public function start(Request $request)
    {
        $data = $request->validate(['invite_code_id' => 'required|integer']);

        $invite = InviteCode::query()->with('exam')->whereKey($data['invite_code_id'])->firstOrFail();

        if ($invite->used_at || now()->greaterThan($invite->expires_at) || !$invite->exam->is_active) {
            abort(403);
        }

        $settings = AntiCheatSetting::query()->first();

        return DB::transaction(function () use ($invite, $request, $settings) {
            // lock one-time code
            $invite->used_at = now();
            $invite->used_ip = $request->ip();
            $invite->used_user_agent = substr((string)$request->userAgent(), 0, 500);
            $invite->save();

            $attempt = Attempt::create([
                'exam_id' => $invite->exam_id,
                'student_id' => $invite->student_id,
                'invite_code_id' => $invite->id,
                'started_at' => now(),
                'ip' => $request->ip(),
                'user_agent' => substr((string)$request->userAgent(), 0, 500),
                'tab_switch_count' => 0,
                'exam_title_snapshot' => $invite->exam->title,
                'student_name_snapshot' => $invite->student_snapshot_name,
                'student_code_snapshot' => $invite->student_snapshot_code,
                'code_snapshot' => $invite->code,
                'status' => 'OK',
            ]);

            $questions = $invite->exam->questions()
                ->whereNull('deleted_at')
                ->where('is_active', true)
                ->get();

            // shuffle if enabled
            if ($settings && $settings->shuffle_questions) {
                $questions = $questions->shuffle()->values();
            }

            return view('student.take', [
                'attempt' => $attempt,
                'exam' => $invite->exam,
                'questions' => $questions,
                'settings' => $settings,
            ]);
        });
    }

    public function submit(Request $request)
    {
        $data = $request->validate([
            'attempt_id' => 'required|integer',
            'answers' => 'required|array',
            'tab_switch_count' => 'nullable|integer|min:0',
        ]);

        $attempt = Attempt::query()->with(['exam.questions'])->whereKey($data['attempt_id'])->firstOrFail();
        if ($attempt->finished_at) abort(403);

        $settings = AntiCheatSetting::query()->first();

        return DB::transaction(function () use ($attempt, $data, $settings) {
            $questions = $attempt->exam?->questions()
                ->whereNull('deleted_at')
                ->where('is_active', true)
                ->get() ?? collect();

            $total = 0;
            $earned = 0;

            foreach ($questions as $q) {
                $total += (int)$q->score;

                $selected = isset($data['answers'][$q->id]) ? (int)$data['answers'][$q->id] : 0;
                $isCorrect = ((int)$selected === (int)$q->correct_index);
                $scoreAwarded = $isCorrect ? (int)$q->score : 0;
                $earned += $scoreAwarded;

                Answer::create([
                    'attempt_id' => $attempt->id,
                    'question_id' => $q->id,
                    'question_text_snapshot' => $q->question_text,
                    'choices_snapshot' => $q->choices,
                    'correct_index_snapshot' => (int)$q->correct_index,
                    'selected_index' => $selected,
                    'is_correct' => $isCorrect,
                    'score_awarded' => $scoreAwarded,
                ]);
            }

            $attempt->finished_at = now();
            $attempt->score = $earned;
            $attempt->score_percent = $total > 0 ? round(($earned / $total) * 100, 2) : 0.0;
            if (isset($data['tab_switch_count'])) {
                $attempt->tab_switch_count = (int)$data['tab_switch_count'];
            }

            // basic anti-cheat flag
            if ($settings && $settings->prevent_tab_switch && $attempt->tab_switch_count > $settings->max_tab_switch_allowed) {
                $attempt->status = 'FLAGGED';
            }

            $attempt->save();

            return view('student.done', ['attempt' => $attempt]);
        });
    }
}
